// This project is released under the GPL Licence

// Compressor, with variable order envelopes.

/*****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <math.h>

/*****************************************************************************/

#include "ladspa.h"

/*****************************************************************************/

/* The port numbers for the plugin: */

#define PXU_CONTROL1 0
#define PXU_CONTROL2 1
#define PXU_CONTROL3 2

#define PXU_INPUT1  3
#define PXU_OUTPUT1 4
#define PXU_INPUT2  5
#define PXU_OUTPUT2 6

/*****************************************************************************/
	
	
/* The structure used to hold port connection information and state
   (actually gain controls require no further state). */

typedef struct {

  /* Ports:
     ------ */
  LADSPA_Data * m_pfControlValue1;
  LADSPA_Data * m_pfControlValue2;
  LADSPA_Data * m_pfControlValue3;
  LADSPA_Data * m_pfInputBuffer1;
  LADSPA_Data * m_pfOutputBuffer1;
  LADSPA_Data * m_pfInputBuffer2;  /* (Not used for mono) */
  LADSPA_Data * m_pfOutputBuffer2; /* (Not used for mono) */

  double b_sr;
  double b_off;
  double b_lenv;
  double b_sensitivity;
  double b_gse1buf;
  double b_gse2buf;
  double b_gse3buf;
  double b_gse4buf;
  double b_gse5buf; 
  
} Pxu;


/*****************************************************************************/


LADSPA_Handle 
instantiatePxu(const LADSPA_Descriptor * Descriptor,
			   unsigned long             SampleRate) {

  Pxu * psPxu;

  psPxu 
    = (Pxu *)malloc(sizeof(Pxu));

  if (psPxu == NULL) 
    return NULL;
  
    psPxu->b_sr = (LADSPA_Data)SampleRate;
    psPxu->b_lenv = 1;
    psPxu->b_sensitivity = 0.001;
    psPxu->b_off = 14.430;
    psPxu->b_gse1buf = 1;
    psPxu->b_gse2buf = 1;
    psPxu->b_gse3buf = 1;
    psPxu->b_gse4buf = 1;
    psPxu->b_gse5buf = 1; 
    psPxu->b_sensitivity = pow(2,(((0.8570)*-1000))/ 6);
  
  return psPxu;
}


/*****************************************************************************/

/* Connect a port to a data location. */
void 
connectPortToPxu(LADSPA_Handle Instance,
		       unsigned long Port,
		       LADSPA_Data * DataLocation) {

  Pxu * psPxu;

  psPxu = (Pxu *)Instance;
  switch (Port) {
  case PXU_CONTROL1:
    psPxu->m_pfControlValue1 = DataLocation;  
    break;
  case PXU_CONTROL2:
    psPxu->m_pfControlValue2 = DataLocation; 
    break;
  case PXU_CONTROL3:
    psPxu->m_pfControlValue3 = DataLocation; 
    break;
  case PXU_INPUT1:
    psPxu->m_pfInputBuffer1 = DataLocation;
    break;
  case PXU_OUTPUT1:
    psPxu->m_pfOutputBuffer1 = DataLocation;
    break;
  case PXU_INPUT2:
    /* (This should only happen for stereo.) */
    psPxu->m_pfInputBuffer2 = DataLocation;
    break;
  case PXU_OUTPUT2:
    /* (This should only happen for stereo.) */
    psPxu->m_pfOutputBuffer2 = DataLocation;
    break;
  }

}

/*****************************************************************************/

void 
runMonoPxu(LADSPA_Handle Instance,
		 unsigned long SampleCount) {
  
  LADSPA_Data * pfInput;
  LADSPA_Data * pfOutput;
  LADSPA_Data fVar1;
  LADSPA_Data fVar2;
  LADSPA_Data fVar3;
  Pxu * psPxu;
  unsigned long lSampleIndex;

  psPxu = (Pxu *)Instance;

  pfInput = psPxu->m_pfInputBuffer1;
  pfOutput = psPxu->m_pfOutputBuffer1;
  fVar1 = *(psPxu->m_pfControlValue1);
  fVar2 = *(psPxu->m_pfControlValue2);
  fVar3 = *(psPxu->m_pfControlValue3);

  for (lSampleIndex = 0; lSampleIndex < SampleCount; lSampleIndex++) 
    *(pfOutput++) = *(pfInput++) * fVar1;
}

/*****************************************************************************/

void 
runStereoPxu(LADSPA_Handle Instance,
		   unsigned long SampleCount) {
  
  LADSPA_Data * pfInput1;
  LADSPA_Data * pfInput2;
  LADSPA_Data * pfOutput1;
  LADSPA_Data * pfOutput2;
  LADSPA_Data fVar1;
  LADSPA_Data fVar2;
  LADSPA_Data fVar3;
  
  Pxu * psPxu;
  unsigned long lSampleIndex;

  psPxu = (Pxu *)Instance;
  
  fVar1 = *(psPxu->m_pfControlValue1);
  fVar2 = *(psPxu->m_pfControlValue2);
  fVar3 = *(psPxu->m_pfControlValue3);
  
  pfInput1 = psPxu->m_pfInputBuffer1;
  pfInput2 = psPxu->m_pfInputBuffer2;
  pfOutput1 = psPxu->m_pfOutputBuffer1;
  pfOutput2 = psPxu->m_pfOutputBuffer2;
  

  // adjust to samplerate 
  double b_sr = psPxu->b_sr;                   
  double b_srt = b_sr/44100;

  // front
  double b_ingain = pow(2,(((fVar1-0.5)*100))/ 6); 
  double b_response = (pow(2,(((1-(fVar3))*-20))/ 1)) / b_srt;  
//  double b_response = pow(fVar3,6);
 
 
// Fast Gaussian filter (variable 1th to 9nth order lowpass, minimal phase)
// response
		double b_gse1gain;
		double b_gse2gain;
		double b_gse3gain;
		double b_gse4gain;
		double b_gse5gain;

		b_gse1gain = 1;
		b_gse2gain = -4;
		b_gse3gain = 6;
		b_gse4gain = -4;
		b_gse5gain = 1;

		double b_gse1cut;
		double b_gse2cut;
		double b_gse3cut;
		double b_gse4cut;
		double b_gse5cut;

		double b_gseslope = 1-pow(2,(((0.5000)*-100))/ 6); 
		double b_gsecut = 1-b_response;
		b_gsecut = b_gsecut / b_srt; // adjust to samplerate.

		b_gse1cut = b_gsecut * b_gseslope * b_gseslope * b_gseslope * b_gseslope;
		b_gse2cut = b_gsecut * b_gseslope * b_gseslope * b_gseslope;
		b_gse3cut = b_gsecut * b_gseslope * b_gseslope;
		b_gse4cut = b_gsecut * b_gseslope;
		b_gse5cut = b_gsecut;

		b_gse1cut = 1-b_gse1cut;
		b_gse2cut = 1-b_gse2cut;
		b_gse3cut = 1-b_gse3cut;
		b_gse4cut = 1-b_gse4cut;
		b_gse5cut = 1-b_gse5cut;

		b_gse1gain = b_gse1gain / b_gse1cut;
		b_gse2gain = b_gse2gain / b_gse2cut;
		b_gse3gain = b_gse3gain / b_gse3cut;
		b_gse4gain = b_gse4gain / b_gse4cut;
		b_gse5gain = b_gse5gain / b_gse5cut;

//kahan here aswell.
double b_cErr;
double b_cSum;
double b_cSumpre;
double b_cSub;
double b_cIn;

double b_of1;
double b_of2;
double b_of3;
double b_of4;
double b_of5;

double b_gsenrm = 0;

		b_of1 = b_gse1gain;
		b_of2 = b_gse2gain;
		b_of3 = b_gse3gain;
		b_of4 = b_gse4gain;
		b_of5 = b_gse5gain;
		
		b_cErr = 0;
		b_cSum = 0;
		b_cSumpre = 0;
		b_cSub = 0;
		b_cIn = 0;

	b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
	b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
		b_gsenrm = 1/b_cSum;


		b_gse1gain = b_gse1gain * b_gsenrm;
		b_gse2gain = b_gse2gain * b_gsenrm;
		b_gse3gain = b_gse3gain * b_gsenrm;
		b_gse4gain = b_gse4gain * b_gsenrm;
		b_gse5gain = b_gse5gain * b_gsenrm;

//////////
		
  for (lSampleIndex = 0; lSampleIndex < SampleCount; lSampleIndex++) 
 {
   double b_in1 = *(pfInput1++); 
   double b_in2 = *(pfInput2++); 

		b_in1 = b_in1 * b_ingain;
		b_in2 = b_in2 * b_ingain;
		
		// Rectification and SH for envelope and detector.
		double b_inrr;
		double b_inr = b_in1; if (b_inr < 0) {b_inr = -b_inr;}
		double b_inr2 = b_in2; if (b_inr2 < 0) {b_inr2 = -b_inr2;}
		if (b_inr2 > b_inr) {b_inr = b_inr2;}
		
		
//		if (b_inr < psPxu->b_sensitivity) {b_inr = psPxu->b_sensitivity;} // sensitivity and zero removal..

		// kneed attack
//		psPxu->b_lenv = b_inr / psPxu->b_lenv;
//		double b_knee = b_inr; if (b_knee > 1) {b_knee = 1;}
//		if (b_knee < psPxu->b_lenv) {psPxu->b_lenv = b_knee;}
//		psPxu->b_lenv = b_inr / psPxu->b_lenv;

		
		psPxu->b_lenv = b_inr; if (psPxu->b_lenv < 1) {psPxu->b_lenv = 1;}

		// Gaussian Env Smoother / Denoiser / Saturated Attack "ubkgauss59"
//		psPxu->b_lenv = psPxu->b_lenv*psPxu->b_lenv; // rms

		psPxu->b_gse1buf = psPxu->b_gse1buf + ((-psPxu->b_gse1buf + psPxu->b_lenv) * b_gse1cut); // onepoles.
		psPxu->b_gse2buf = psPxu->b_gse2buf + ((-psPxu->b_gse2buf + psPxu->b_lenv) * b_gse2cut);
		psPxu->b_gse3buf = psPxu->b_gse3buf + ((-psPxu->b_gse3buf + psPxu->b_lenv) * b_gse3cut);
		psPxu->b_gse4buf = psPxu->b_gse4buf + ((-psPxu->b_gse4buf + psPxu->b_lenv) * b_gse4cut);
		psPxu->b_gse5buf = psPxu->b_gse5buf + ((-psPxu->b_gse5buf + psPxu->b_lenv) * b_gse5cut);

		b_of1 = b_gse1gain * psPxu->b_gse1buf;
		b_of2 = b_gse2gain * psPxu->b_gse2buf;
		b_of3 = b_gse3gain * psPxu->b_gse3buf;
		b_of4 = b_gse4gain * psPxu->b_gse4buf;
		b_of5 = b_gse5gain * psPxu->b_gse5buf;

		b_cErr = 0;
		b_cSum = 0;
		b_cSumpre = 0;
		b_cSub = 0;
		b_cIn = 0;

	b_cIn = b_of1;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
	b_cIn = b_of2;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of3;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of4;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;

	b_cIn = b_of5;
		b_cSub = b_cIn - b_cErr; b_cSumpre = b_cSum;
		b_cSum = b_cSum + b_cSub;
		b_cErr = (b_cSum-b_cSumpre) - b_cSub;
		
	psPxu->b_lenv = b_cSum;

//	psPxu->b_lenv = b_of1+b_of2+b_of3+b_of4+b_of5;
//psPxu->b_lenv = psPxu->b_gse1buf;
//		psPxu->b_lenv = sqrt(psPxu->b_lenv);
		
		b_in1 = b_in1 / psPxu->b_lenv;
		b_in2 = b_in2 / psPxu->b_lenv;


   *(pfOutput1++) = b_in1;
   *(pfOutput2++) = b_in2;
 }


}

/*****************************************************************************/

/* Throw away a simple delay line. */
void 
cleanupPxu(LADSPA_Handle Instance) {
  free(Instance);
}

/*****************************************************************************/

LADSPA_Descriptor * g_psMonoDescriptor = NULL;
LADSPA_Descriptor * g_psStereoDescriptor = NULL;

/*****************************************************************************/

/* _init() is called automatically when the plugin library is first
   loaded. */
void 
_init() {

  char ** pcPortNames;
  LADSPA_PortDescriptor * piPortDescriptors;
  LADSPA_PortRangeHint * psPortRangeHints;

  g_psMonoDescriptor
    = (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));
  g_psStereoDescriptor 
    = (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

  if (g_psMonoDescriptor) {
  
    g_psMonoDescriptor->UniqueID
      = 990;                                         	
    g_psMonoDescriptor->Label
      = strdup("Mlm_Compressor_M");				
    g_psMonoDescriptor->Properties
      = LADSPA_PROPERTY_HARD_RT_CAPABLE;
    g_psMonoDescriptor->Name 
      = strdup("Millennium Compressor [mono] (not implemented)");			
    g_psMonoDescriptor->Maker
      = strdup("Opensource GPL");
    g_psMonoDescriptor->Copyright
      = strdup("Opensource GPL");
    g_psMonoDescriptor->PortCount
      = 5;						
    piPortDescriptors
      = (LADSPA_PortDescriptor *)calloc(5, sizeof(LADSPA_PortDescriptor));
    g_psMonoDescriptor->PortDescriptors
      = (const LADSPA_PortDescriptor *)piPortDescriptors;
    piPortDescriptors[PXU_CONTROL1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL3]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_INPUT1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT1]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    pcPortNames
      = (char **)calloc(5, sizeof(char *));
    g_psMonoDescriptor->PortNames 
      = (const char **)pcPortNames;
    pcPortNames[PXU_CONTROL1]
      = strdup("1");
    pcPortNames[PXU_CONTROL2]
      = strdup("2");
    pcPortNames[PXU_CONTROL3]
      = strdup("3");
    pcPortNames[PXU_INPUT1]
      = strdup("Input");
    pcPortNames[PXU_OUTPUT1]
      = strdup("Output");
    psPortRangeHints = ((LADSPA_PortRangeHint *)
			calloc(5, sizeof(LADSPA_PortRangeHint)));
    g_psMonoDescriptor->PortRangeHints
      = (const LADSPA_PortRangeHint *)psPortRangeHints;
    psPortRangeHints[PXU_CONTROL1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_CONTROL2].HintDescriptor
      = 0;
    psPortRangeHints[PXU_CONTROL3].HintDescriptor
      = 0;
    psPortRangeHints[PXU_INPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT1].HintDescriptor
      = 0;
    g_psMonoDescriptor->instantiate 
      = instantiatePxu;
    g_psMonoDescriptor->connect_port 
      = connectPortToPxu;
    g_psMonoDescriptor->activate
      = NULL;
    g_psMonoDescriptor->run
      = runMonoPxu;
    g_psMonoDescriptor->run_adding
      = NULL;
    g_psMonoDescriptor->set_run_adding_gain
      = NULL;
    g_psMonoDescriptor->deactivate
      = NULL;
    g_psMonoDescriptor->cleanup
      = cleanupPxu;
  }
  
  if (g_psStereoDescriptor) {
    
    g_psStereoDescriptor->UniqueID
      = 991;
    g_psStereoDescriptor->Label
      = strdup("Mlm_Compressor_S");
    g_psStereoDescriptor->Properties
      = LADSPA_PROPERTY_HARD_RT_CAPABLE;
    g_psStereoDescriptor->Name 
      = strdup("Millennium Compressor [stereo]");
    g_psStereoDescriptor->Maker
      = strdup("Opensource GPL)");
    g_psStereoDescriptor->Copyright
      = strdup("Opensource GPL");
    g_psStereoDescriptor->PortCount
      = 7;
    piPortDescriptors
      = (LADSPA_PortDescriptor *)calloc(7, sizeof(LADSPA_PortDescriptor));
    g_psStereoDescriptor->PortDescriptors
      = (const LADSPA_PortDescriptor *)piPortDescriptors;
    piPortDescriptors[PXU_CONTROL1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL3]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_INPUT1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT1]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_INPUT2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT2]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    pcPortNames
      = (char **)calloc(7, sizeof(char *));
    g_psStereoDescriptor->PortNames 
      = (const char **)pcPortNames;
    pcPortNames[PXU_CONTROL1]
      = strdup("Threshold");
    pcPortNames[PXU_CONTROL2]
      = strdup("RMS");
    pcPortNames[PXU_CONTROL3]
      = strdup("Response");
    pcPortNames[PXU_INPUT1]
      = strdup("Input (Left)");
    pcPortNames[PXU_OUTPUT1]
      = strdup("Output (Left)");
    pcPortNames[PXU_INPUT2]
      = strdup("Input (Right)");
    pcPortNames[PXU_OUTPUT2]
      = strdup("Output (Right)");
    psPortRangeHints = ((LADSPA_PortRangeHint *)
			calloc(7, sizeof(LADSPA_PortRangeHint)));
    g_psStereoDescriptor->PortRangeHints
      = (const LADSPA_PortRangeHint *)psPortRangeHints;

    psPortRangeHints[PXU_CONTROL1].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL1].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL1].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL2].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL2].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL2].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL3].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL3].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL3].UpperBound
      = 1;

    psPortRangeHints[PXU_INPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_INPUT2].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT2].HintDescriptor
      = 0;
    g_psStereoDescriptor->instantiate 
      = instantiatePxu;
    g_psStereoDescriptor->connect_port 
      = connectPortToPxu;
    g_psStereoDescriptor->activate
      = NULL;
    g_psStereoDescriptor->run
      = runStereoPxu;
    g_psStereoDescriptor->run_adding
      = NULL;
    g_psStereoDescriptor->set_run_adding_gain
      = NULL;
    g_psStereoDescriptor->deactivate
      = NULL;
    g_psStereoDescriptor->cleanup
      = cleanupPxu;
  }
}

/*****************************************************************************/

void
deleteDescriptor(LADSPA_Descriptor * psDescriptor) {
  unsigned long lIndex;
  if (psDescriptor) {
    free((char *)psDescriptor->Label);
    free((char *)psDescriptor->Name);
    free((char *)psDescriptor->Maker);
    free((char *)psDescriptor->Copyright);
    free((LADSPA_PortDescriptor *)psDescriptor->PortDescriptors);
    for (lIndex = 0; lIndex < psDescriptor->PortCount; lIndex++)
      free((char *)(psDescriptor->PortNames[lIndex]));
    free((char **)psDescriptor->PortNames);
    free((LADSPA_PortRangeHint *)psDescriptor->PortRangeHints);
    free(psDescriptor);
  }
}

/*****************************************************************************/

/* _fini() is called automatically when the library is unloaded. */
void
_fini() {
  deleteDescriptor(g_psMonoDescriptor);
  deleteDescriptor(g_psStereoDescriptor);
}

/*****************************************************************************/

/* Return a descriptor of the requested plugin type. There are two
   plugin types available in this library (mono and stereo). */
const LADSPA_Descriptor * 
ladspa_descriptor(unsigned long Index) {
  /* Return the requested descriptor or null if the index is out of
     range. */
  switch (Index) {
  case 0:
    return g_psMonoDescriptor;
  case 1:
    return g_psStereoDescriptor;
  default:
    return NULL;
  }
}

/*****************************************************************************/

/* EOF */
